{-----------------------------------------------------------------------------
This Software is placed into Publlic Domain and distributed on an "AS IS" basis,
WITHOUT WARRANTY OF ANY KIND, either express or implied.
The Initial Developer is Martin Waldenburg
(Martin.Waldenburg@T-Online.de).
Contributor: James Jacobson
-----------------------------------------------------------------------------}
unit mwDelphiDirectiveParser;

interface

uses
  SysUtils, mwDelphiLanguageElements,
  mwD4NLexer;

type
  TmwDelphiDirectiveParser = class(TmwD4NLexer)
  protected
    function CheckCondition: Boolean;
    function CheckDeclared: Boolean;
    function CheckDefined: Boolean;
    function CheckDirectiveExpression: Boolean;
    function CheckTypeOfConstant: Integer;
    function CheckTypeOfRelation: Integer;
    procedure CheckCommaOrEnd;
    function CheckIfDirective: Boolean;
    procedure DirectiveHandler;
    function EvaluateDirectiveExpression(ConstTypeOne: Integer; ConstOne: AnsiString;
      RelationType, ConstTypeTwo: Integer; ConstTwo: AnsiString): Boolean; virtual;
    procedure FindDirectiveEnd;
    procedure IncludeFile(FileName: AnsiString);
    procedure LinkObjectFile(FileName: AnsiString);
    procedure MainDirectiveHandler;
    procedure leAHandler;
    procedure leBHandler;
    procedure leCHandler;
    procedure leDHandler;
    procedure leEHandler;
    procedure leGHandler;
    procedure leHHandler;
    procedure leIHandler;
    procedure leJHandler;
    procedure leLHandler;
    procedure leMHandler;
    procedure leOHandler;
    procedure lePHandler;
    procedure leQHandler;
    procedure leRHandler;
    procedure leSHandler;
    procedure leTHandler;
    procedure leUHandler;
    procedure leVHandler;
    procedure leWHandler;
    procedure leXHandler;
    procedure leYHandler;
    procedure leZHandler;
    procedure leYDHandler;
    procedure leIfDirectiveHandler;
    procedure leIfDefHandler;
    procedure leEndIfHandler;
    procedure leIfEndHandler;
    procedure leElseDirectiveHandler;
    procedure leDefineHandler;
    procedure leAlignHandler;
    procedure leIfNDefHandler;
    procedure leLinkHandler;
    procedure leUnDefHandler;
    procedure leElseIfHandler;
    procedure leImageBaseHandler;
    procedure leIfOptHandler;
    procedure leSoNameHandler;
    procedure leIncludeHandler;
    procedure leHintsHandler;
    procedure leNoDefineHandler;
    procedure leIOChecksHandler;
    procedure leDebugInfoHandler;
    procedure leBoolevalHandler;
    procedure leHPPEmitHandler;
    procedure leRangeChecksHandler;
    procedure leNoIncludeHandler;
    procedure leApptypeHandler;
    procedure leSafeDivideHandler;
    procedure leStackChecksHandler;
    procedure leResourceHandler;
    procedure leWarnHandler;
    procedure leWarningsHandler;
    procedure leTypeInfoHandler;
    procedure leSoPrefixHandler;
    procedure leStackFramesHandler;
    procedure leRunOnlyHandler;
    procedure leSoSuffixHandler;
    procedure leReferenceInfoHandler;
    procedure leDesignOnlyHandler;
    procedure leExtensionHandler;
    procedure leImportedDataHandler;
    procedure leDescriptionHandler;
    procedure leSoVersionHandler;
    procedure leAssertionsHandler;
    procedure leImplicitBuildHandler;
    procedure leTypedAddressHandler;
    procedure leLocalSymbolsHandler;
    procedure leMinEnumSizeHandler;
    procedure leWeakPackageUnitHandler;
    procedure leDefinitionInfoHandler;
    procedure leMinStackSizeHandler;
    procedure leObjExportAllHandler;
    procedure leMaxStackSizeHandler;
    procedure leLongStringsHandler;
    procedure leDenyPackageUnitHandler;
    procedure leExternalSymHandler;
    procedure leOpenStringsHandler;
    procedure leOverFlowChecksHandler;
    procedure leWriteableConstHandler;
    procedure leOptimizationHandler;
    procedure leVarStringChecksHandler;
    procedure leExtendedSyntaxHandler;
    procedure leRealCompatibilityHandler;
    procedure leResourceReserveHandler;
    procedure leRegionHandler;
    procedure leEndRegionHandler;
    procedure SkipUntilElseOrEndIf;
    procedure SkipUntilElseIfOrIfEnd;
    procedure SkipUntilEndIf;
    procedure SkipUntilIfEnd;
  public
    procedure NextToken; override;
    procedure Parse; virtual;
  end;

implementation

procedure TmwDelphiDirectiveParser.CheckCommaOrEnd;
var
  TempRange: TmwDelphiRange;
begin
  TempRange := Range;
  Next;
  case Id of
    leComma:
      begin
        Next;
        DirectiveHandler;
      end;
    leCurlyClose:
      case TempRange of
        drBorlandDirective: Id := leBorlandDirective;
      else Id := leUnknownDirective;
      end;
    leEndOfAnsiComment:
      case TempRange of
        drAnsiDirective: Id := leAnsiDirective;
      else Id := leUnknownDirective;
      end;
  else FindDirectiveEnd;
  end;
end;

function TmwDelphiDirectiveParser.CheckDeclared: Boolean;
begin
  Result := False;
  NextNoSpace;
  if Id = leRoundOpen then
    NextNoSpace;
  case (Id = leIdentifier) or (ExId = leIdentifier) of
    True:
      begin
        Result := fDeclaredList.IndexOf(Token) > -1;
      end;
  end;
  NextNoSpace;
  if Id = leRoundClose then
    NextNoSpace;
end;

function TmwDelphiDirectiveParser.CheckDefined: Boolean;
begin
  Result := False;
  NextNoSpace;
  if Id = leRoundOpen then
    NextNoSpace;
  case (Id = leIdentifier) or (ExId = leIdentifier) of
    True:
      begin
        Result := fDefinedList.IndexOf(Token) > -1;
      end;
  end;
  NextNoSpace;
  if Id = leRoundClose then
    NextNoSpace;
end;

function TmwDelphiDirectiveParser.CheckTypeOfConstant: Integer;
begin
  Result := leUnknown;
  case Id of
    leNumber, leFloat, leStringConstant:
      Result := Id;
  else
    if (Id = leIdentifier) or (ExId = leIdentifier) then Result := leIdentifier;
  end;
end;

function TmwDelphiDirectiveParser.CheckTypeOfRelation: Integer;
begin
  Result := leUnknown;
  case Id of
    leEqual, leNotEqual, leLess, leGreater, leLessOrEqual, leGreaterOrEqual:
      Result := ID;
  end;
end;

function TmwDelphiDirectiveParser.CheckDirectiveExpression: Boolean;
var
  ConstOne, ConstTwo: string;
  ConstTypeOne, RelationType, ConstTypeTwo: Integer;
begin
  Result := False;
  case Id of
    leRoundOpen:
      begin
        NextNoSpace;
        Result := CheckCondition;
        if id = leRoundClose then
          NextNoSpace;
        case Id of
          leAnd:
            begin
              NextNoSpace;
              Result := Result and CheckCondition;
            end;
          leOr:
            begin
              NextNoSpace;
              Result := Result or CheckCondition;
            end;
        end;
      end;
  else
    begin
      ConstTypeOne := CheckTypeOfConstant;
      ConstOne := Token;
      if ConstTypeOne <> leUnknown then
        NextNoSpace;
      RelationType := CheckTypeOfRelation;
      if RelationType <> leUnknown then
        NextNoSpace;
      ConstTypeTwo := CheckTypeOfConstant;
      ConstTwo := Token;
      if (ConstTypeOne <> leUnknown)
        and (ConstTypeTwo <> leUnknown)
        and (RelationType <> leUnknown) then
      begin
        Result := EvaluateDirectiveExpression(ConstTypeOne, ConstOne, RelationType,
          ConstTypeTwo, ConstTwo);
        NextNoSpace;
        case Id of
          leAnd:
            begin
              NextNoSpace;
              Result := Result and CheckCondition;
            end;
          leOr:
            begin
              NextNoSpace;
              Result := Result or CheckCondition;
            end;
        end;
      end;
    end;
  end;
end;

function TmwDelphiDirectiveParser.CheckCondition: Boolean;
begin
  Result := False;
  case Id of
    leAtEnd: ;
    leDeclared:
      begin
        Result := CheckDeclared;
      end;
    leDefined:
      begin
        Result := CheckDefined;
      end;
    leNumber, leFloat, leStringConstant:
      begin
        Result := CheckDirectiveExpression;
      end;
    leRoundOpen: Result := CheckDirectiveExpression;
  else
    begin
      case (Id = leIdentifier) or (ExId = leIdentifier) of
        True: Result := CheckDirectiveExpression;
      end;
    end;
  end;
end;

function TmwDelphiDirectiveParser.CheckIfDirective: Boolean;
begin
  Result := CheckCondition;
end;

procedure TmwDelphiDirectiveParser.DirectiveHandler;
begin
  case Id of
    leA: leAHandler;
    leB: leBHandler;
    leC: leCHandler;
    leD: leDHandler;
    leE: leEHandler;
    leG: leGHandler;
    leH: leHHandler;
    leI: leIHandler;
    leJ: leJHandler;
    leL: leLHandler;
    leM: leMHandler;
    leO: leOHandler;
    leP: lePHandler;
    leQ: leQHandler;
    leR: leRHandler;
    leS: leSHandler;
    leT: leTHandler;
    leU: leUHandler;
    leV: leVHandler;
    leW: leWHandler;
    leX: leXHandler;
    leY: leYHandler;
    leZ: leZHandler;
    leYD: leYDHandler;
    leDefine: leDefineHandler;
    leAlign: leAlignHandler;
    leIfNDef: leIfNDefHandler;
    leLink: leLinkHandler;
    leUnDef: leUnDefHandler;
    leImageBase: leImageBaseHandler;
    leSoName: leSoNameHandler;
    leInclude: leIncludeHandler;
    leHints: leHintsHandler;
    leNoDefine: leNoDefineHandler;
    leIOChecks: leIOChecksHandler;
    leDebugInfo: leDebugInfoHandler;
    leBooleval: leBoolevalHandler;
    leHPPEmit: leHPPEmitHandler;
    leRangeChecks: leRangeChecksHandler;
    leNoInclude: leNoIncludeHandler;
    leApptype: leApptypeHandler;
    leSafeDivide: leSafeDivideHandler;
    leResource: leResourceHandler;
    leWarn: leWarnHandler;
    leWarnings: leWarningsHandler;
    leTypeInfo: leTypeInfoHandler;
    leSoPrefix: leSoPrefixHandler;
    leStackFrames: leStackFramesHandler;
    leRunOnly: leRunOnlyHandler;
    leSoSuffix: leSoSuffixHandler;
    leReferenceInfo: leReferenceInfoHandler;
    leDesignOnly: leDesignOnlyHandler;
    leExtension: leExtensionHandler;
    leImportedData: leImportedDataHandler;
    leDescription: leDescriptionHandler;
    leSoVersion: leSoVersionHandler;
    leAssertions: leAssertionsHandler;
    leImplicitBuild: leImplicitBuildHandler;
    leTypedAddress: leTypedAddressHandler;
    leLocalSymbols: leLocalSymbolsHandler;
    leMinEnumSize: leMinEnumSizeHandler;
    leWeakPackageUnit: leWeakPackageUnitHandler;
    leDefinitionInfo: leDefinitionInfoHandler;
    leMinStackSize: leMinStackSizeHandler;
    leObjExportAll: leObjExportAllHandler;
    leMaxStackSize: leMaxStackSizeHandler;
    leLongStrings: leLongStringsHandler;
    leDenyPackageUnit: leDenyPackageUnitHandler;
    leExternalSym: leExternalSymHandler;
    leOpenStrings: leOpenStringsHandler;
    leOverFlowChecks: leOverFlowChecksHandler;
    leWriteableConst: leWriteableConstHandler;
    leOptimization: leOptimizationHandler;
    leVarStringChecks: leVarStringChecksHandler;
    leExtendedSyntax: leExtendedSyntaxHandler;
    leRealCompatibility: leRealCompatibilityHandler;
    leResourceReserve: leResourceReserveHandler;
    leRegion: leRegionHandler;
    leEndRegion: leEndRegionHandler;
  end;
end;

function TmwDelphiDirectiveParser.EvaluateDirectiveExpression(
  ConstTypeOne: Integer; ConstOne: AnsiString; RelationType,
  ConstTypeTwo: Integer; ConstTwo: AnsiString): Boolean;
begin
  // To be implemented in derived classes
  Result := False;
end;

procedure TmwDelphiDirectiveParser.FindDirectiveEnd;
var
  TempRange: TmwDelphiRange;
begin
  TempRange := Range;
  while (Id <> leAtEnd) and (Range <> drNormal) do
    Next;
  case Id of
    leCurlyClose:
      case TempRange of
        drBorlandDirective: Id := leBorlandDirective;
      else Id := leUnknownDirective;
      end;
    leEndOfAnsiComment:
      case TempRange of
        drAnsiDirective: Id := leAnsiDirective;
      else Id := leUnknownDirective;
      end;
  else Id := leUnknownDirective;
  end;
end;

procedure TmwDelphiDirectiveParser.IncludeFile(FileName: AnsiString);
begin

end;

procedure TmwDelphiDirectiveParser.MainDirectiveHandler;
var
  TempStart: Integer;
begin
  TempStart := Start;
  Next;
  case Id of
    leA: leAHandler;
    leB: leBHandler;
    leC: leCHandler;
    leD: leDHandler;
    leE: leEHandler;
    leG: leGHandler;
    leH: leHHandler;
    leI: leIHandler;
    leJ: leJHandler;
    leL: leLHandler;
    leM: leMHandler;
    leO: leOHandler;
    leP: lePHandler;
    leQ: leQHandler;
    leR: leRHandler;
    leS: leSHandler;
    leT: leTHandler;
    leU: leUHandler;
    leV: leVHandler;
    leW: leWHandler;
    leX: leXHandler;
    leY: leYHandler;
    leZ: leZHandler;
    leYD: leYDHandler;
    leIfDirective: leIfDirectiveHandler;
    leIfDef: leIfDefHandler;
    leEndIf: leEndIfHandler;
    leIfEnd: leIfEndHandler;
    leElseDirective: leElseDirectiveHandler;
    leDefine: leDefineHandler;
    leAlign: leAlignHandler;
    leIfNDef: leIfNDefHandler;
    leLink: leLinkHandler;
    leUnDef: leUnDefHandler;
    leElseIf: leElseIfHandler;
    leImageBase: leImageBaseHandler;
    leIfOpt: leIfOptHandler;
    leSoName: leSoNameHandler;
    leInclude: leIncludeHandler;
    leHints: leHintsHandler;
    leNoDefine: leNoDefineHandler;
    leIOChecks: leIOChecksHandler;
    leDebugInfo: leDebugInfoHandler;
    leBooleval: leBoolevalHandler;
    leHPPEmit: leHPPEmitHandler;
    leRangeChecks: leRangeChecksHandler;
    leNoInclude: leNoIncludeHandler;
    leApptype: leApptypeHandler;
    leSafeDivide: leSafeDivideHandler;
    leResource: leResourceHandler;
    leWarn: leWarnHandler;
    leWarnings: leWarningsHandler;
    leTypeInfo: leTypeInfoHandler;
    leSoPrefix: leSoPrefixHandler;
    leStackFrames: leStackFramesHandler;
    leRunOnly: leRunOnlyHandler;
    leSoSuffix: leSoSuffixHandler;
    leReferenceInfo: leReferenceInfoHandler;
    leDesignOnly: leDesignOnlyHandler;
    leExtension: leExtensionHandler;
    leImportedData: leImportedDataHandler;
    leDescription: leDescriptionHandler;
    leSoVersion: leSoVersionHandler;
    leAssertions: leAssertionsHandler;
    leImplicitBuild: leImplicitBuildHandler;
    leTypedAddress: leTypedAddressHandler;
    leLocalSymbols: leLocalSymbolsHandler;
    leMinEnumSize: leMinEnumSizeHandler;
    leWeakPackageUnit: leWeakPackageUnitHandler;
    leDefinitionInfo: leDefinitionInfoHandler;
    leMinStackSize: leMinStackSizeHandler;
    leObjExportAll: leObjExportAllHandler;
    leMaxStackSize: leMaxStackSizeHandler;
    leLongStrings: leLongStringsHandler;
    leDenyPackageUnit: leDenyPackageUnitHandler;
    leExternalSym: leExternalSymHandler;
    leOpenStrings: leOpenStringsHandler;
    leOverFlowChecks: leOverFlowChecksHandler;
    leWriteableConst: leWriteableConstHandler;
    leOptimization: leOptimizationHandler;
    leVarStringChecks: leVarStringChecksHandler;
    leExtendedSyntax: leExtendedSyntaxHandler;
    leRealCompatibility: leRealCompatibilityHandler;
    leResourceReserve: leResourceReserveHandler;
    leRegion: leRegionHandler;
    leEndRegion: leEndRegionHandler;
  end;
  Start := TempStart;
end;

procedure TmwDelphiDirectiveParser.leAHandler;
begin
  Next;
  case Id of
    leMinus: Exclude(fBooleanDirectives, bdAlign);
    leNumber:
      case Run - Start = 1 of
        True:
          case Buf[Start] of
            '1':
              begin
                fRecordAlignment := 1;
                Exclude(fBooleanDirectives, bdAlign);
              end;
            '2':
              begin
                fRecordAlignment := 2;
                Include(fBooleanDirectives, bdAlign);
              end;
            '4':
              begin
                fRecordAlignment := 4;
                Include(fBooleanDirectives, bdAlign);
              end;
            '8':
              begin
                fRecordAlignment := 8;
                Include(fBooleanDirectives, bdAlign);
              end;
          end;
        False: Id := leUnknownDirective;
      end;
    lePlus: Include(fBooleanDirectives, bdAlign);
  else Id := leUnknownDirective;
  end;
  if Id <> leUnknownDirective then
    CheckCommaOrEnd;
end;

procedure TmwDelphiDirectiveParser.leAlignHandler;
begin
  NextNoSpace;
  case Id of
    leNumber:
      case Run - Start = 1 of
        True:
          case Buf[Start] of
            '1':
              begin
                fRecordAlignment := 1;
                Exclude(fBooleanDirectives, bdAlign);
              end;
            '2':
              begin
                fRecordAlignment := 2;
                Include(fBooleanDirectives, bdAlign);
              end;
            '4':
              begin
                fRecordAlignment := 4;
                Include(fBooleanDirectives, bdAlign);
              end;
            '8':
              begin
                fRecordAlignment := 8;
                Include(fBooleanDirectives, bdAlign);
              end;
          end;
        False: Id := leUnknownDirective;
      end;
    leOff: Exclude(fBooleanDirectives, bdAlign);
    leOnDirective: Include(fBooleanDirectives, bdAlign);
  else Id := leUnknownDirective;
  end;
  if Id <> leUnknownDirective then
    CheckCommaOrEnd;
end;

procedure TmwDelphiDirectiveParser.leApptypeHandler;
begin
  NextNoSpace;
  case Id of
    leConsole: fApptype := atConsole;
    leGUI: fApptype := atGUI;
  else Id := leUnknownDirective;
  end;
  if Id <> leUnknownDirective then
    CheckCommaOrEnd;
end;

procedure TmwDelphiDirectiveParser.leAssertionsHandler;
begin
  NextNoSpace;
  case Id of
    leOff: Exclude(fBooleanDirectives, bdAssertions);
    leOnDirective: Include(fBooleanDirectives, bdAssertions);
  else Id := leUnknownDirective;
  end;
  if Id <> leUnknownDirective then
    CheckCommaOrEnd;
end;

procedure TmwDelphiDirectiveParser.leBHandler;
begin
  Next;
  case Id of
    leMinus: Exclude(fBooleanDirectives, bdBoolEval);
    lePlus: Include(fBooleanDirectives, bdBoolEval);
  else Id := leUnknownDirective;
  end;
  if Id <> leUnknownDirective then
    CheckCommaOrEnd;
end;

procedure TmwDelphiDirectiveParser.leBoolevalHandler;
begin
  NextNoSpace;
  case Id of
    leOff: Exclude(fBooleanDirectives, bdBooleval);
    leOnDirective: Include(fBooleanDirectives, bdBooleval);
  else Id := leUnknownDirective;
  end;
  if Id <> leUnknownDirective then
    CheckCommaOrEnd;
end;

procedure TmwDelphiDirectiveParser.leCHandler;
begin
  Next;
  case Id of
    leMinus: Exclude(fBooleanDirectives, bdAssertions);
    lePlus: Include(fBooleanDirectives, bdAssertions);
  else Id := leUnknownDirective;
  end;
  if Id <> leUnknownDirective then
    CheckCommaOrEnd;
end;

procedure TmwDelphiDirectiveParser.leDebugInfoHandler;
begin
  NextNoSpace;
  case Id of
    leOff: Exclude(fBooleanDirectives, bdDebugInfo);
    leOnDirective: Include(fBooleanDirectives, bdDebugInfo);
  else Id := leUnknownDirective;
  end;
  if Id <> leUnknownDirective then
    CheckCommaOrEnd;
end;

procedure TmwDelphiDirectiveParser.leDefineHandler;
begin
  NextNoSpace;
  case (Id = leIdentifier) or (ExId = leIdentifier) of
    True: fDefinedList.Add(Token);
  else Id := leUnknownDirective;
  end;
  if Id <> leUnknownDirective then
    CheckCommaOrEnd;
end;

procedure TmwDelphiDirectiveParser.leDefinitionInfoHandler;
begin
  NextNoSpace;
  case Id of
    leOff: Exclude(fBooleanDirectives, bdDefinitionInfo);
    leOnDirective: Include(fBooleanDirectives, bdDefinitionInfo);
  else Id := leUnknownDirective;
  end;
  if Id <> leUnknownDirective then
    CheckCommaOrEnd;
end;

procedure TmwDelphiDirectiveParser.leDenyPackageUnitHandler;
begin
  NextNoSpace;
  case Id of
    leOff: Exclude(fBooleanDirectives, bdDenyPackageUnit);
    leOnDirective: Include(fBooleanDirectives, bdDenyPackageUnit);
  else Id := leUnknownDirective;
  end;
  if Id <> leUnknownDirective then
    CheckCommaOrEnd;
end;

procedure TmwDelphiDirectiveParser.leDescriptionHandler;
var
  Temp: AnsiString;
begin
  NextNoSpace;
  case Id of
    leStringConstant:
      begin
        SetLength(Temp, Run - Start - 2);
{$IFDEF CLR}
        System.array.Copy(Buf, Start + 1, Temp, 0, Run - Start - 2);
{$ELSE}
        System.Move(Buf[Start + 1], Temp[1], Run - Start - 2);
{$ENDIF}
        SetDescription(Temp);
        CheckCommaOrEnd;
      end;
  else Id := leUnknownDirective;
  end;
end;

procedure TmwDelphiDirectiveParser.leDesignOnlyHandler;
begin
  NextNoSpace;
  case Id of
    leOff: Exclude(fBooleanDirectives, bdDesignOnly);
    leOnDirective: Include(fBooleanDirectives, bdDesignOnly);
  else Id := leUnknownDirective;
  end;
  if Id <> leUnknownDirective then
    CheckCommaOrEnd;
end;

procedure TmwDelphiDirectiveParser.leDHandler;
begin
  Next;
  case Id of
    leMinus: Exclude(fBooleanDirectives, bdDebugInfo);
    lePlus: Include(fBooleanDirectives, bdDebugInfo);
  else Id := leUnknownDirective;
  end;
  if Id <> leUnknownDirective then
    CheckCommaOrEnd;
end;

procedure TmwDelphiDirectiveParser.leEHandler;
begin
  NextNoSpace;
  case (Id = leIdentifier) or (ExId = leIdentifier) of
    True: SetExtension(Token);
  else Id := leUnknownDirective;
  end;
  if Id <> leUnknownDirective then
  begin
    CheckCommaOrEnd;
  end;
end;

procedure TmwDelphiDirectiveParser.leElseDirectiveHandler;
var
  Condition: Boolean;
begin
  FindDirectiveEnd;
  Condition := False;
  if Id <> leUnknownDirective then
  begin
    Id := leElseDirective;
    if DirectiveStack.Count > 0 then
      case Integer(DirectiveStack.Peek) of
        leIfDef, leIfNDef, leIfOpt,
        leElseIf:
          Condition := True;
      end;
    case Condition of
      True:
        begin
          DirectiveStack.Pop;
          DirectiveStack.Push(TObject(Integer(leElseDirective)));
          Next;
          SkipUntilEndIf;
        end;
      False:
        begin
          if Assigned(fOnLexerError) then
            fOnLexerError(Self, 'Else directive without previous leIfDef, leIfNDef or leIfOpt directive');
        end;
    end;
  end;
end;

procedure TmwDelphiDirectiveParser.leElseIfHandler;
var
  Condition: Boolean;
begin
  FindDirectiveEnd;
  Condition := False;
  if Id <> leUnknownDirective then
  begin
    Id := leElseIf;
    if DirectiveStack.Count > 0 then
      case Integer(DirectiveStack.Peek) of
        leIfDirective: Condition := True;
      end;
    case Condition of
      True:
        begin
          DirectiveStack.Pop;
          DirectiveStack.Push(TObject(Integer(leElseIf)));
          Next;
          SkipUntilIfEnd;
        end;
      False:
        begin
          if Assigned(fOnLexerError) then
            fOnLexerError(Self, 'ElseIf without previous If directive');
        end;
    end;
  end;
end;

procedure TmwDelphiDirectiveParser.leEndIfHandler;
var
  Condition: Boolean;
begin
  FindDirectiveEnd;
  Condition := False;
  if Id <> leUnknownDirective then
  begin
    id := leEndIf;
    if DirectiveStack.Count > 0 then
      case Integer(DirectiveStack.Peek) of
        leElseDirective, leIfDef, leIfNDef, leIfOpt: Condition := True;
      end;
    case Condition of
      True: DirectiveStack.Pop;
      False:
        begin
          if Assigned(fOnLexerError) then
            fOnLexerError(Self, 'EndIf without previous IfDef, Else, IfNDef or IfOpt directive');
        end;
    end;
  end;
end;

procedure TmwDelphiDirectiveParser.leEndRegionHandler;
begin
  FindDirectiveEnd;
  if Id <> leUnknownDirective then
  begin
    id := leEndRegion;
  end;
end;

procedure TmwDelphiDirectiveParser.leExtendedSyntaxHandler;
begin
  NextNoSpace;
  case Id of
    leOff: Exclude(fBooleanDirectives, bdExtendedSyntax);
    leOnDirective: Include(fBooleanDirectives, bdExtendedSyntax);
  else Id := leUnknownDirective;
  end;
  if Id <> leUnknownDirective then
    CheckCommaOrEnd;
end;

procedure TmwDelphiDirectiveParser.leExtensionHandler;
begin
  NextNoSpace;
  case (Id = leIdentifier) or (ExId = leIdentifier) of
    True: SetExtension(Token);
  else Id := leUnknownDirective;
  end;
  if Id <> leUnknownDirective then
  begin
    CheckCommaOrEnd;
  end;
end;

procedure TmwDelphiDirectiveParser.leExternalSymHandler;
begin
  NextNoSpace;
  case (Id = leIdentifier) or (ExId = leIdentifier) of
    True: SetExternalSym(Token);
  else Id := leUnknownDirective;
  end;
  if Id <> leUnknownDirective then
  begin
    CheckCommaOrEnd;
  end;
end;

procedure TmwDelphiDirectiveParser.leGHandler;
begin
  Next;
  case Id of
    leMinus: Exclude(fBooleanDirectives, bdImportedData);
    lePlus: Include(fBooleanDirectives, bdImportedData);
  else Id := leUnknownDirective;
  end;
  if Id <> leUnknownDirective then
    CheckCommaOrEnd;
end;

procedure TmwDelphiDirectiveParser.leHHandler;
begin
  Next;
  case Id of
    leMinus: Exclude(fBooleanDirectives, bdLongStrings);
    lePlus: Include(fBooleanDirectives, bdLongStrings);
  else Id := leUnknownDirective;
  end;
  if Id <> leUnknownDirective then
    CheckCommaOrEnd;
end;

procedure TmwDelphiDirectiveParser.leHintsHandler;
begin
  NextNoSpace;
  case Id of
    leOff: Exclude(fBooleanDirectives, bdHints);
    leOnDirective: Include(fBooleanDirectives, bdHints);
  else Id := leUnknownDirective;
  end;
  if Id <> leUnknownDirective then
    CheckCommaOrEnd;
end;

procedure TmwDelphiDirectiveParser.leHPPEmitHandler;
var
  Temp: AnsiString;
begin
  NextNoSpace;
  case Id of
    leStringConstant:
      begin
        if fEmitForHpp then
        begin
          SetLength(Temp, Run - Start - 2);
{$IFDEF CLR}
          System.array.Copy(Buf, Start + 1, Temp, 0, Run - Start - 2);
{$ELSE}
          System.Move(Buf[Start + 1], Temp[1], Run - Start - 2);
{$ENDIF}
          SetHppEmit(Temp);
        end;
        CheckCommaOrEnd;
      end;
  else Id := leUnknownDirective;
  end;
end;

procedure TmwDelphiDirectiveParser.leIfDefHandler;
var
  IsDefined: Boolean;
begin
  NextNoSpace;
  case (Id = leIdentifier) or (ExId = leIdentifier) of
    True:
      begin
        IsDefined := DefinedList.IndexOf(Token) > -1;
        FindDirectiveEnd;
        Id := leIfDef;
        DirectiveStack.Push(TObject(Integer(leIfDef)));
        case IsDefined of
          False:
            begin
              Next;
              SkipUntilElseOrEndIf;
            end;
        end;
      end;
  else Id := leUnknownDirective;
  end;
end;

procedure TmwDelphiDirectiveParser.leIfDirectiveHandler;
var
  Condition: Boolean;
begin
  Condition := CheckIfDirective;
  FindDirectiveEnd;
  case Condition of
    False:
      begin
        DirectiveStack.Push(TObject(Integer(leIfDirective)));
        Next;
        SkipUntilElseIfOrIfEnd;
      end;
  end;
end;

procedure TmwDelphiDirectiveParser.leIfEndHandler;
var
  Condition: Boolean;
begin
  FindDirectiveEnd;
  Condition := False;
  if Id <> leUnknownDirective then
  begin
    id := leIfEnd;
    if DirectiveStack.Count > 0 then
      case Integer(DirectiveStack.Peek) of
        leElseIf, leIfDirective,
        leElseDirective:
          Condition := True;
      end;
    case Condition of
      True: DirectiveStack.Pop;
      False:
        begin
          if Assigned(fOnLexerError) then
            fOnLexerError(Self, 'IfEnd without previous If or ElseIf directive');
        end;
    end;
  end;
end;

procedure TmwDelphiDirectiveParser.leIfNDefHandler;
var
  IsDefined: Boolean;
begin
  NextNoSpace;
  case (Id = leIdentifier) or (ExId = leIdentifier) of
    True:
      begin
        IsDefined := DefinedList.IndexOf(Token) > -1;
        FindDirectiveEnd;
        Id := leIfNDef;
        DirectiveStack.Push(TObject(Integer(leIfNDef)));
        case IsDefined of
          True:
            begin
              Next;
              SkipUntilElseOrEndIf;
            end;
        end;
      end;
  else Id := leUnknownDirective;
  end;
end;

procedure TmwDelphiDirectiveParser.leIfOptHandler;
var
  Condition: Boolean;
begin
  Condition := False;
  NextNoSpace;
  case Id of
    leA:
      begin
        Next;
        case Id of
          leMinus: Condition := not (bdAlign in fBooleanDirectives);
          lePlus: Condition := bdAlign in fBooleanDirectives;
        else Id := leUnknownDirective;
        end;
      end;
    leB:
      begin
        Next;
        case Id of
          leMinus: Condition := not (bdBoolEval in fBooleanDirectives);
          lePlus: Condition := bdBoolEval in fBooleanDirectives;
        else Id := leUnknownDirective;
        end;
      end;
    leC:
      begin
        Next;
        case Id of
          leMinus: Condition := not (bdAssertions in fBooleanDirectives);
          lePlus: Condition := bdAssertions in fBooleanDirectives;
        else Id := leUnknownDirective;
        end;
      end;
    leD:
      begin
        Next;
        case Id of
          leMinus: Condition := not (bdDebugInfo in fBooleanDirectives);
          lePlus: Condition := bdDebugInfo in fBooleanDirectives;
        else Id := leUnknownDirective;
        end;
      end;
    leG:
      begin
        Next;
        case Id of
          leMinus: Condition := not (bdImportedData in fBooleanDirectives);
          lePlus: Condition := bdImportedData in fBooleanDirectives;
        else Id := leUnknownDirective;
        end;
      end;
    leH:
      begin
        Next;
        case Id of
          leMinus: Condition := not (bdLongStrings in fBooleanDirectives);
          lePlus: Condition := bdLongStrings in fBooleanDirectives;
        else Id := leUnknownDirective;
        end;
      end;
    leI:
      begin
        Next;
        case Id of
          leMinus: Condition := not (bdIoChecks in fBooleanDirectives);
          lePlus: Condition := bdIoChecks in fBooleanDirectives;
        else Id := leUnknownDirective;
        end;
      end;
    leJ:
      begin
        Next;
        case Id of
          leMinus: Condition := not (bdWriteAbleConst in fBooleanDirectives);
          lePlus: Condition := bdWriteAbleConst in fBooleanDirectives;
        else Id := leUnknownDirective;
        end;
      end;
    leL:
      begin
        Next;
        case Id of
          leMinus: Condition := not (bdLocalSymbols in fBooleanDirectives);
          lePlus: Condition := bdLocalSymbols in fBooleanDirectives;
        else Id := leUnknownDirective;
        end;
      end;
    leM:
      begin
        Next;
        case Id of
          leMinus: Condition := not (bdTypeInfo in fBooleanDirectives);
          lePlus: Condition := bdTypeInfo in fBooleanDirectives;
        else Id := leUnknownDirective;
        end;
      end;
    leO:
      begin
        Next;
        case Id of
          leMinus: Condition := not (bdOptimization in fBooleanDirectives);
          lePlus: Condition := bdOptimization in fBooleanDirectives;
        else Id := leUnknownDirective;
        end;
      end;
    leP:
      begin
        Next;
        case Id of
          leMinus: Condition := not (bdOpenStrings in fBooleanDirectives);
          lePlus: Condition := bdOpenStrings in fBooleanDirectives;
        else Id := leUnknownDirective;
        end;
      end;
    leQ:
      begin
        Next;
        case Id of
          leMinus: Condition := not (bdOverflowChecks in fBooleanDirectives);
          lePlus: Condition := bdOverflowChecks in fBooleanDirectives;
        else Id := leUnknownDirective;
        end;
      end;
    leR:
      begin
        Next;
        case Id of
          leMinus: Condition := not (bdRangeChecks in fBooleanDirectives);
          lePlus: Condition := bdRangeChecks in fBooleanDirectives;
        else Id := leUnknownDirective;
        end;
      end;
    leS:
      begin
        Next;
        case Id of
          leMinus: Condition := not (bdStackChecks in fBooleanDirectives);
          lePlus: Condition := bdStackChecks in fBooleanDirectives;
        else Id := leUnknownDirective;
        end;
      end;
    leT:
      begin
        Next;
        case Id of
          leMinus: Condition := not (bdTypedAddress in fBooleanDirectives);
          lePlus: Condition := bdTypedAddress in fBooleanDirectives;
        else Id := leUnknownDirective;
        end;
      end;
    leU:
      begin
        Next;
        case Id of
          leMinus: Condition := not (bdSafeDivide in fBooleanDirectives);
          lePlus: Condition := bdSafeDivide in fBooleanDirectives;
        else Id := leUnknownDirective;
        end;
      end;
    leV:
      begin
        Next;
        case Id of
          leMinus: Condition := not (bdVarStringChecks in fBooleanDirectives);
          lePlus: Condition := bdVarStringChecks in fBooleanDirectives;
        else Id := leUnknownDirective;
        end;
      end;
    leW:
      begin
        Next;
        case Id of
          leMinus: Condition := not (bdStackFrames in fBooleanDirectives);
          lePlus: Condition := bdStackFrames in fBooleanDirectives;
        else Id := leUnknownDirective;
        end;
      end;
    leX:
      begin
        Next;
        case Id of
          leMinus: Condition := not (bdExtendedSyntax in fBooleanDirectives);
          lePlus: Condition := bdExtendedSyntax in fBooleanDirectives;
        else Id := leUnknownDirective;
        end;
      end;
    leY:
      begin
        Next;
        case Id of
          leMinus: Condition := not (bdDefinitionInfo in fBooleanDirectives);
          lePlus: Condition := bdDefinitionInfo in fBooleanDirectives;
        else Id := leUnknownDirective;
        end;
      end;
    leYD:
      begin
        FindDirectiveEnd;
        case Id <> leUnknownDirective of
          True: Condition := bdDefinitionInfo in fBooleanDirectives;
          False: Id := leUnknownDirective;
        end;
      end;
    leAlign:
      begin
        NextNoSpace;
        case Id of
          leOff: Condition := not (bdAlign in fBooleanDirectives);
          leOnDirective: Condition := bdAlign in fBooleanDirectives;
        else Id := leUnknownDirective;
        end;
      end;
    leAssertions:
      begin
        NextNoSpace;
        case Id of
          leOff: Condition := not (bdAssertions in fBooleanDirectives);
          leOnDirective: Condition := bdAssertions in fBooleanDirectives;
        else Id := leUnknownDirective;
        end;
      end;
    leBooleval:
      begin
        NextNoSpace;
        case Id of
          leOff: Condition := not (bdBooleval in fBooleanDirectives);
          leOnDirective: Condition := bdBooleval in fBooleanDirectives;
        else Id := leUnknownDirective;
        end;
      end;
    leDebugInfo:
      begin
        NextNoSpace;
        case Id of
          leOff: Condition := not (bdDebugInfo in fBooleanDirectives);
          leOnDirective: Condition := bdDebugInfo in fBooleanDirectives;
        else Id := leUnknownDirective;
        end;
      end;
    leDenyPackageUnit:
      begin
        NextNoSpace;
        case Id of
          leOff: Condition := not (bdDenyPackageUnit in fBooleanDirectives);
          leOnDirective: Condition := bdDenyPackageUnit in fBooleanDirectives;
        else Id := leUnknownDirective;
        end;
      end;
    leDesignOnly:
      begin
        NextNoSpace;
        case Id of
          leOff: Condition := not (bdDesignOnly in fBooleanDirectives);
          leOnDirective: Condition := bdDesignOnly in fBooleanDirectives;
        else Id := leUnknownDirective;
        end;
      end;
    leObjExportAll:
      begin
        NextNoSpace;
        case Id of
          leOff: Condition := not (bdObjExportAll in fBooleanDirectives);
          leOnDirective: Condition := bdObjExportAll in fBooleanDirectives;
        else Id := leUnknownDirective;
        end;
      end;
    leExtendedSyntax:
      begin
        NextNoSpace;
        case Id of
          leOff: Condition := not (bdExtendedSyntax in fBooleanDirectives);
          leOnDirective: Condition := bdExtendedSyntax in fBooleanDirectives;
        else Id := leUnknownDirective;
        end;
      end;
    leHints:
      begin
        NextNoSpace;
        case Id of
          leOff: Condition := not (bdHints in fBooleanDirectives);
          leOnDirective: Condition := bdHints in fBooleanDirectives;
        else Id := leUnknownDirective;
        end;
      end;
    leImplicitBuild:
      begin
        NextNoSpace;
        case Id of
          leOff: Condition := not (bdImplicitBuild in fBooleanDirectives);
          leOnDirective: Condition := bdImplicitBuild in fBooleanDirectives;
        else Id := leUnknownDirective;
        end;
      end;
    leImportedData:
      begin
        NextNoSpace;
        case Id of
          leOff: Condition := not (bdImportedData in fBooleanDirectives);
          leOnDirective: Condition := bdImportedData in fBooleanDirectives;
        else Id := leUnknownDirective;
        end;
      end;
    leIOChecks:
      begin
        NextNoSpace;
        case Id of
          leOff: Condition := not (bdIOChecks in fBooleanDirectives);
          leOnDirective: Condition := bdIOChecks in fBooleanDirectives;
        else Id := leUnknownDirective;
        end;
      end;
    leLocalSymbols:
      begin
        NextNoSpace;
        case Id of
          leOff: Condition := not (bdLocalSymbols in fBooleanDirectives);
          leOnDirective: Condition := bdLocalSymbols in fBooleanDirectives;
        else Id := leUnknownDirective;
        end;
      end;
    leLongStrings:
      begin
        NextNoSpace;
        case Id of
          leOff: Condition := not (bdLongStrings in fBooleanDirectives);
          leOnDirective: Condition := bdLongStrings in fBooleanDirectives;
        else Id := leUnknownDirective;
        end;
      end;
    leOpenStrings:
      begin
        NextNoSpace;
        case Id of
          leOff: Condition := not (bdOpenStrings in fBooleanDirectives);
          leOnDirective: Condition := bdOpenStrings in fBooleanDirectives;
        else Id := leUnknownDirective;
        end;
      end;
    leOptimization:
      begin
        NextNoSpace;
        case Id of
          leOff: Condition := not (bdOptimization in fBooleanDirectives);
          leOnDirective: Condition := bdOptimization in fBooleanDirectives;
        else Id := leUnknownDirective;
        end;
      end;
    leOverFlowChecks:
      begin
        NextNoSpace;
        case Id of
          leOff: Condition := not (bdOverFlowChecks in fBooleanDirectives);
          leOnDirective: Condition := bdOverFlowChecks in fBooleanDirectives;
        else Id := leUnknownDirective;
        end;
      end;
    leRangeChecks:
      begin
        NextNoSpace;
        case Id of
          leOff: Condition := not (bdRangeChecks in fBooleanDirectives);
          leOnDirective: Condition := bdRangeChecks in fBooleanDirectives;
        else Id := leUnknownDirective;
        end;
      end;
    leRealCompatibility:
      begin
        NextNoSpace;
        case Id of
          leOff: Condition := not (bdRealCompatibility in fBooleanDirectives);
          leOnDirective: Condition := bdRealCompatibility in fBooleanDirectives;
        else Id := leUnknownDirective;
        end;
      end;
    leRunOnly:
      begin
        NextNoSpace;
        case Id of
          leOff: Condition := not (bdRunOnly in fBooleanDirectives);
          leOnDirective: Condition := bdRunOnly in fBooleanDirectives;
        else Id := leUnknownDirective;
        end;
      end;
    leStackChecks:
      begin
        NextNoSpace;
        case Id of
          leOff: Condition := not (bdStackChecks in fBooleanDirectives);
          leOnDirective: Condition := bdStackChecks in fBooleanDirectives;
        else Id := leUnknownDirective;
        end;
      end;
    leTypeInfo:
      begin
        NextNoSpace;
        case Id of
          leOff: Condition := not (bdTypeInfo in fBooleanDirectives);
          leOnDirective: Condition := bdTypeInfo in fBooleanDirectives;
        else Id := leUnknownDirective;
        end;
      end;
    leDefinitionInfo:
      begin
        NextNoSpace;
        case Id of
          leOff: Condition := not (bdDefinitionInfo in fBooleanDirectives);
          leOnDirective: Condition := bdDefinitionInfo in fBooleanDirectives;
        else Id := leUnknownDirective;
        end;
      end;
    leSafeDivide:
      begin
        NextNoSpace;
        case Id of
          leOff: Condition := not (bdSafeDivide in fBooleanDirectives);
          leOnDirective: Condition := bdSafeDivide in fBooleanDirectives;
        else Id := leUnknownDirective;
        end;
      end;
    leTypedAddress:
      begin
        NextNoSpace;
        case Id of
          leOff: Condition := not (bdTypedAddress in fBooleanDirectives);
          leOnDirective: Condition := bdTypedAddress in fBooleanDirectives;
        else Id := leUnknownDirective;
        end;
      end;
    leVarStringChecks:
      begin
        NextNoSpace;
        case Id of
          leOff: Condition := not (bdVarStringChecks in fBooleanDirectives);
          leOnDirective: Condition := bdVarStringChecks in fBooleanDirectives;
        else Id := leUnknownDirective;
        end;
      end;
    leWarnings:
      begin
        NextNoSpace;
        case Id of
          leOff: Condition := not (bdWarnings in fBooleanDirectives);
          leOnDirective: Condition := bdWarnings in fBooleanDirectives;
        else Id := leUnknownDirective;
        end;
      end;
    leWeakPackageUnit:
      begin
        NextNoSpace;
        case Id of
          leOff: Condition := not (bdWeakPackageUnit in fBooleanDirectives);
          leOnDirective: Condition := bdWeakPackageUnit in fBooleanDirectives;
        else Id := leUnknownDirective;
        end;
      end;
    leStackFrames:
      begin
        NextNoSpace;
        case Id of
          leOff: Condition := not (bdStackFrames in fBooleanDirectives);
          leOnDirective: Condition := bdStackFrames in fBooleanDirectives;
        else Id := leUnknownDirective;
        end;
      end;
    leWriteableConst:
      begin
        NextNoSpace;
        case Id of
          leOff: Condition := not (bdWriteableConst in fBooleanDirectives);
          leOnDirective: Condition := bdWriteableConst in fBooleanDirectives;
        else Id := leUnknownDirective;
        end;
      end;
  else Id := leUnknownDirective;
  end;
  if Id <> leUnknownDirective then
  begin
    FindDirectiveEnd;
    Id := leIfOpt;
    DirectiveStack.Push(TObject(Integer(leIfOpt)));
    if Condition = False then
    begin
      Next;
      SkipUntilElseOrEndIf;
    end;
  end;
end;

procedure TmwDelphiDirectiveParser.leIHandler;
var
  Temp: AnsiString;
begin
  Next;
  case Id of
    leMinus: Exclude(fBooleanDirectives, bdIOChecks);
    lePlus: Include(fBooleanDirectives, bdIOChecks);
    leSpace:
      begin
        NextNoSpace;
        case Id of
          leIdentifier: Temp := RetrieveIncludeFileName;
          leStringConstant:
            begin
              SetLength(Temp, Run - Start - 2);
{$IFDEF CLR}
              System.array.Copy(Buf, Start + 1, Temp, 0, Run - Start - 2);
{$ELSE}
              System.Move(Buf[Start + 1], Temp[1], Run - Start - 2);
{$ENDIF}
            end;
        else
          case ExId = leIdentifier of
            True: Temp := RetrieveIncludeFileName;
            False: Id := leUnknownDirective;
          end;
        end;
      end;
  else Id := leUnknownDirective;
  end;
  if Id <> leUnknownDirective then
    CheckCommaOrEnd;
  if Id <> leUnknownDirective then
    IncludeFile(Temp);
end;

procedure TmwDelphiDirectiveParser.leImageBaseHandler;
begin
  NextNoSpace;
  case Id of
    leHexNumber,
      leNumber: SetImageBase(StrToInt(Token));
  else Id := leUnknownDirective;
  end;
  if Id <> leUnknownDirective then
    CheckCommaOrEnd;
end;

procedure TmwDelphiDirectiveParser.leImplicitBuildHandler;
begin
  NextNoSpace;
  case Id of
    leOff: Exclude(fBooleanDirectives, bdImplicitBuild);
    leOnDirective: Include(fBooleanDirectives, bdImplicitBuild);
  else Id := leUnknownDirective;
  end;
  if Id <> leUnknownDirective then
    CheckCommaOrEnd;
end;

procedure TmwDelphiDirectiveParser.leImportedDataHandler;
begin
  NextNoSpace;
  case Id of
    leOff: Exclude(fBooleanDirectives, bdImportedData);
    leOnDirective: Include(fBooleanDirectives, bdImportedData);
  else Id := leUnknownDirective;
  end;
  if Id <> leUnknownDirective then
    CheckCommaOrEnd;
end;

procedure TmwDelphiDirectiveParser.leIncludeHandler;
var
  Temp: AnsiString;
begin
  NextNoSpace;
  case Id of
    leIdentifier: Temp := RetrieveIncludeFileName;
    leStringConstant:
      begin
        SetLength(Temp, Run - Start - 2);
{$IFDEF CLR}
        System.array.Copy(Buf, Start + 1, Temp, 0, Run - Start - 2);
{$ELSE}
        System.Move(Buf[Start + 1], Temp[1], Run - Start - 2);
{$ENDIF}
      end;
  else
    case ExId = leIdentifier of
      True: Temp := RetrieveIncludeFileName;
      False: Id := leUnknownDirective;
    end;
  end;
  if Id <> leUnknownDirective then
    CheckCommaOrEnd;
  if Id <> leUnknownDirective then
    IncludeFile(Temp);
end;

procedure TmwDelphiDirectiveParser.leIOChecksHandler;
begin
  NextNoSpace;
  case Id of
    leOf: Exclude(fBooleanDirectives, bdIOChecks);
    leOnDirective: Include(fBooleanDirectives, bdIOChecks);
  else Id := leUnknownDirective;
  end;
  if Id <> leUnknownDirective then
    CheckCommaOrEnd;
end;

procedure TmwDelphiDirectiveParser.leJHandler;
begin
  Next;
  case Id of
    leMinus: Exclude(fBooleanDirectives, bdWriteableConst);
    lePlus: Include(fBooleanDirectives, bdWriteableConst);
  else Id := leUnknownDirective;
  end;
  if Id <> leUnknownDirective then
    CheckCommaOrEnd;
end;

procedure TmwDelphiDirectiveParser.leLHandler;
begin
  Next;
  case Id of
    leMinus: Exclude(fBooleanDirectives, bdLocalSymbols);
    lePlus: Include(fBooleanDirectives, bdLocalSymbols);
    leSpace:
      begin
        NextNoSpace;
        case (Id = leIdentifier) or (ExId = leIdentifier) of
          True: LinkObjectFile(Token);
        else Id := leUnknownDirective;
        end;
      end;
  else Id := leUnknownDirective;
  end;
  if Id <> leUnknownDirective then
    CheckCommaOrEnd;
end;

procedure TmwDelphiDirectiveParser.leLinkHandler;
begin
  NextNoSpace;
  case (Id = leIdentifier) or (ExId = leIdentifier) of
    True: LinkObjectFile(Token);
  else Id := leUnknownDirective;
  end;
  if Id <> leUnknownDirective then
    CheckCommaOrEnd;
end;

procedure TmwDelphiDirectiveParser.leLocalSymbolsHandler;
begin
  NextNoSpace;
  case Id of
    leOff: Exclude(fBooleanDirectives, bdLocalSymbols);
    leOnDirective: Include(fBooleanDirectives, bdLocalSymbols);
  else Id := leUnknownDirective;
  end;
  if Id <> leUnknownDirective then
    CheckCommaOrEnd;
end;

procedure TmwDelphiDirectiveParser.leLongStringsHandler;
begin
  NextNoSpace;
  case Id of
    leOff: Exclude(fBooleanDirectives, bdLongStrings);
    leOnDirective: Include(fBooleanDirectives, bdLongStrings);
  else Id := leUnknownDirective;
  end;
  if Id <> leUnknownDirective then
    CheckCommaOrEnd;
end;

procedure TmwDelphiDirectiveParser.leMaxStackSizeHandler;
begin
  NextNoSpace;
  case (Id = leHexNumber) or (Id = leNumber) of
    True: SetMaxStackSize(StrToInt(Token));
  else Id := leUnknownDirective;
  end;
  if Id <> leUnknownDirective then
    CheckCommaOrEnd;
end;

procedure TmwDelphiDirectiveParser.leMHandler;
var
  Temp: AnsiString;
  TempStart, TempRun: Integer;
begin
  Next;
  case Id of
    leMinus: Exclude(fBooleanDirectives, bdTypeInfo);
    lePlus: Include(fBooleanDirectives, bdTypeInfo);
    leSpace:
      begin
        NextNoSpace;
        case (Id = leIdentifier) or (ExId = leIdentifier) or
          (Id = leHexNumber) or (Id = leNumber) of
          True:
            begin
              Temp := Token;
              TempStart := Start;
              TempRun := Run;
              NextNoSpace;
              case Id = leComma of
                True:
                  begin
                    NextNoSpace;
                    case (Id = leIdentifier) or (ExId = leIdentifier) or
                      (Id = leHexNumber) or (Id = leNumber) of
                      True:
                        begin
                          SetMinStackSize(StrToInt(Temp));
                          SetMaxStackSize(StrToInt(Token));
                        end;
                    else Id := leUnknownDirective;
                    end;
                  end;
                False:
                  begin
                    Start := TempStart;
                    Run := TempRun;
                    SetResourceReserve(Temp);
                  end;
              end;
            end;
        else Id := leUnknownDirective;
        end;
      end;
  else Id := leUnknownDirective;
  end;
  if Id <> leUnknownDirective then
    CheckCommaOrEnd;
end;

procedure TmwDelphiDirectiveParser.leMinEnumSizeHandler;
var
  Temp: Integer;
begin
  NextNoSpace;
  case (Id = leNumber) of
    True:
      begin
        Temp := StrToInt(Token);
        case Temp of
          1, 2, 4: SetMinEnumSize(Temp);
        else Id := leUnknownDirective;
        end;
      end;
  else Id := leUnknownDirective;
  end;
  if Id <> leUnknownDirective then
  begin
    CheckCommaOrEnd;
  end;
end;

procedure TmwDelphiDirectiveParser.leMinStackSizeHandler;
begin
  NextNoSpace;
  case (Id = leHexNumber) or (Id = leNumber) of
    True: SetMinStackSize(StrToInt(Token));
  else Id := leUnknownDirective;
  end;
  if Id <> leUnknownDirective then
    CheckCommaOrEnd;
end;

procedure TmwDelphiDirectiveParser.leNoDefineHandler;
begin
  NextNoSpace;
  case (Id = leIdentifier) or (ExId = leIdentifier) of
    True:
      if fEmitForHpp then
        SetNoDefine(Token);
  else Id := leUnknownDirective;
  end;
  if Id <> leUnknownDirective then
    CheckCommaOrEnd;
end;

procedure TmwDelphiDirectiveParser.leNoIncludeHandler;
begin
  NextNoSpace;
  case (Id = leIdentifier) or (ExId = leIdentifier) of
    True:
      if fEmitForHpp then
        SetNoInclude(Token);
  else Id := leUnknownDirective;
  end;
  if Id <> leUnknownDirective then
    CheckCommaOrEnd;
end;

procedure TmwDelphiDirectiveParser.leObjExportAllHandler;
begin
  NextNoSpace;
  case Id of
    leOff: Exclude(fBooleanDirectives, bdObjExportAll);
    leOnDirective: Include(fBooleanDirectives, bdObjExportAll);
  else Id := leUnknownDirective;
  end;
  if Id <> leUnknownDirective then
    CheckCommaOrEnd;
end;

procedure TmwDelphiDirectiveParser.leOHandler;
begin
  Next;
  case Id of
    leMinus: Exclude(fBooleanDirectives, bdOptimization);
    lePlus: Include(fBooleanDirectives, bdOptimization);
  else Id := leUnknownDirective;
  end;
  if Id <> leUnknownDirective then
    CheckCommaOrEnd;
end;

procedure TmwDelphiDirectiveParser.leOpenStringsHandler;
begin
  NextNoSpace;
  case Id of
    leOff: Exclude(fBooleanDirectives, bdOpenStrings);
    leOnDirective: Include(fBooleanDirectives, bdOpenStrings);
  else Id := leUnknownDirective;
  end;
  if Id <> leUnknownDirective then
    CheckCommaOrEnd;
end;

procedure TmwDelphiDirectiveParser.leOptimizationHandler;
begin
  NextNoSpace;
  case Id of
    leOff: Exclude(fBooleanDirectives, bdOptimization);
    leOnDirective: Include(fBooleanDirectives, bdOptimization);
  else Id := leUnknownDirective;
  end;
  if Id <> leUnknownDirective then
    CheckCommaOrEnd;
end;

procedure TmwDelphiDirectiveParser.leOverFlowChecksHandler;
begin
  NextNoSpace;
  case Id of
    leOff: Exclude(fBooleanDirectives, bdOverFlowChecks);
    leOnDirective: Include(fBooleanDirectives, bdOverFlowChecks);
  else Id := leUnknownDirective;
  end;
  if Id <> leUnknownDirective then
    CheckCommaOrEnd;
end;

procedure TmwDelphiDirectiveParser.lePHandler;
begin
  Next;
  case Id of
    leMinus: Exclude(fBooleanDirectives, bdOpenStrings);
    lePlus: Include(fBooleanDirectives, bdOpenStrings);
  else Id := leUnknownDirective;
  end;
  if Id <> leUnknownDirective then
    CheckCommaOrEnd;
end;

procedure TmwDelphiDirectiveParser.leQHandler;
begin
  Next;
  case Id of
    leMinus: Exclude(fBooleanDirectives, bdOverFlowChecks);
    lePlus: Include(fBooleanDirectives, bdOverFlowChecks);
  else Id := leUnknownDirective;
  end;
  if Id <> leUnknownDirective then
    CheckCommaOrEnd;
end;

procedure TmwDelphiDirectiveParser.leRangeChecksHandler;
begin
  NextNoSpace;
  case Id of
    leOff: Exclude(fBooleanDirectives, bdRangeChecks);
    leOnDirective: Include(fBooleanDirectives, bdRangeChecks);
  else Id := leUnknownDirective;
  end;
  if Id <> leUnknownDirective then
    CheckCommaOrEnd;
end;

procedure TmwDelphiDirectiveParser.leRealCompatibilityHandler;
begin
  NextNoSpace;
  case Id of
    leOff: Exclude(fBooleanDirectives, bdRealCompatibility);
    leOnDirective: Include(fBooleanDirectives, bdRealCompatibility);
  else Id := leUnknownDirective;
  end;
  if Id <> leUnknownDirective then
    CheckCommaOrEnd;
end;

procedure TmwDelphiDirectiveParser.leReferenceInfoHandler;
begin
  NextNoSpace;
  case Id of
    leOff: Exclude(fBooleanDirectives, bdDefinitionInfo);
    leOnDirective: Include(fBooleanDirectives, bdDefinitionInfo);
  else Id := leUnknownDirective;
  end;
  if Id <> leUnknownDirective then
    CheckCommaOrEnd;
end;

procedure TmwDelphiDirectiveParser.leRegionHandler;
begin
  NextNoSpace;
  case Id of
    leStringConstant: SetRegionString(Token);
  else Id := leUnknownDirective;
  end;
  if Id <> leUnknownDirective then
    CheckCommaOrEnd;
end;

procedure TmwDelphiDirectiveParser.leResourceHandler;
begin
  NextNoSpace;
  SetResFileName(RetrieveResourceFileName);
  NextNoSpace;
  SetRcFileName(RetrieveResourceFileName);
  CheckCommaOrEnd;
end;

procedure TmwDelphiDirectiveParser.leResourceReserveHandler;
begin
  NextNoSpace;
  case (Id = leIdentifier) or (ExId = leIdentifier) or
    (Id = leHexNumber) or (Id = leNumber) of
    True: SetResourceReserve(Token);
  else Id := leUnknownDirective;
  end;
  if Id <> leUnknownDirective then
    CheckCommaOrEnd;
end;

procedure TmwDelphiDirectiveParser.leRHandler;
begin
  Next;
  case Id of
    leMinus: Exclude(fBooleanDirectives, bdRangeChecks);
    lePlus: Include(fBooleanDirectives, bdRangeChecks);
    leSpace:
      begin
        NextNoSpace;
        SetResFileName(RetrieveResourceFileName);
        NextNoSpace;
        SetRcFileName(RetrieveResourceFileName);
      end;
  else Id := leUnknownDirective;
  end;
  if Id <> leUnknownDirective then
    CheckCommaOrEnd;
end;

procedure TmwDelphiDirectiveParser.leRunOnlyHandler;
begin
  NextNoSpace;
  case Id of
    leOff: Exclude(fBooleanDirectives, bdRunOnly);
    leOnDirective: Include(fBooleanDirectives, bdRunOnly);
  else Id := leUnknownDirective;
  end;
  if Id <> leUnknownDirective then
    CheckCommaOrEnd;
end;

procedure TmwDelphiDirectiveParser.leSafeDivideHandler;
begin
  NextNoSpace;
  case Id of
    leOff: Exclude(fBooleanDirectives, bdSafeDivide);
    leOnDirective: Include(fBooleanDirectives, bdSafeDivide);
  else Id := leUnknownDirective;
  end;
  if Id <> leUnknownDirective then
    CheckCommaOrEnd;
end;

procedure TmwDelphiDirectiveParser.leSHandler;
begin
  Next;
  case Id of
    leMinus: Exclude(fBooleanDirectives, bdStackChecks);
    lePlus: Include(fBooleanDirectives, bdStackChecks);
  else Id := leUnknownDirective;
  end;
  if Id <> leUnknownDirective then
    CheckCommaOrEnd;
end;

procedure TmwDelphiDirectiveParser.leSoNameHandler;
begin
  //  ToDo
end;

procedure TmwDelphiDirectiveParser.leSoPrefixHandler;
begin
  //  ToDo
end;

procedure TmwDelphiDirectiveParser.leSoSuffixHandler;
begin
  //  ToDo
end;

procedure TmwDelphiDirectiveParser.leSoVersionHandler;
begin
  //  ToDo
end;

procedure TmwDelphiDirectiveParser.leStackChecksHandler;
begin
  NextNoSpace;
  case Id of
    leOf: Exclude(fBooleanDirectives, bdStackChecks);
    leOnDirective: Include(fBooleanDirectives, bdStackChecks);
  else Id := leUnknownDirective;
  end;
  if Id <> leUnknownDirective then
    CheckCommaOrEnd;
end;

procedure TmwDelphiDirectiveParser.leStackFramesHandler;
begin
  NextNoSpace;
  case Id of
    leOf: Exclude(fBooleanDirectives, bdStackFrames);
    leOnDirective: Include(fBooleanDirectives, bdStackFrames);
  else Id := leUnknownDirective;
  end;
  if Id <> leUnknownDirective then
    CheckCommaOrEnd;
end;

procedure TmwDelphiDirectiveParser.leTHandler;
begin
  Next;
  case Id of
    leMinus: Exclude(fBooleanDirectives, bdTypedAddress);
    lePlus: Include(fBooleanDirectives, bdTypedAddress);
  else Id := leUnknownDirective;
  end;
  if Id <> leUnknownDirective then
    CheckCommaOrEnd;
end;

procedure TmwDelphiDirectiveParser.leTypedAddressHandler;
begin
  NextNoSpace;
  case Id of
    leOf: Exclude(fBooleanDirectives, bdTypedAddress);
    leOnDirective: Include(fBooleanDirectives, bdTypedAddress);
  else Id := leUnknownDirective;
  end;
  if Id <> leUnknownDirective then
    CheckCommaOrEnd;
end;

procedure TmwDelphiDirectiveParser.leTypeInfoHandler;
begin
  NextNoSpace;
  case Id of
    leOf: Exclude(fBooleanDirectives, bdTypeInfo);
    leOnDirective: Include(fBooleanDirectives, bdTypeInfo);
  else Id := leUnknownDirective;
  end;
  if Id <> leUnknownDirective then
    CheckCommaOrEnd;
end;

procedure TmwDelphiDirectiveParser.leUHandler;
begin
  Next;
  case Id of
    leMinus: Exclude(fBooleanDirectives, bdSafeDivide);
    lePlus: Include(fBooleanDirectives, bdSafeDivide);
  else Id := leUnknownDirective;
  end;
  if Id <> leUnknownDirective then
    CheckCommaOrEnd;
end;

procedure TmwDelphiDirectiveParser.leUnDefHandler;
begin
  NextNoSpace;
  case (Id = leIdentifier) or (ExId = leIdentifier) of
    True: fDefinedList.Remove(Token);
  else Id := leUnknownDirective;
  end;
  if Id <> leUnknownDirective then
    CheckCommaOrEnd;
end;

procedure TmwDelphiDirectiveParser.leVarStringChecksHandler;
begin
  NextNoSpace;
  case Id of
    leOf: Exclude(fBooleanDirectives, bdVarStringChecks);
    leOnDirective: Include(fBooleanDirectives, bdVarStringChecks);
  else Id := leUnknownDirective;
  end;
  if Id <> leUnknownDirective then
    CheckCommaOrEnd;
end;

procedure TmwDelphiDirectiveParser.leVHandler;
begin
  Next;
  case Id of
    leMinus: Exclude(fBooleanDirectives, bdVarStringChecks);
    lePlus: Include(fBooleanDirectives, bdVarStringChecks);
  else Id := leUnknownDirective;
  end;
  if Id <> leUnknownDirective then
    CheckCommaOrEnd;
end;

procedure TmwDelphiDirectiveParser.leWarnHandler;
begin
  NextNoSpace;
  case Id of
    leOf: Exclude(fBooleanDirectives, bdWarnings);
    leOnDirective: Include(fBooleanDirectives, bdWarnings);
    leSymbol_Platform:
      begin
        NextNoSpace;
        case Id of
          leOf: Exclude(fBooleanDirectives, bdSymbol_Platform);
          leOnDirective: Include(fBooleanDirectives, bdSymbol_Platform);
        else Id := leUnknownDirective;
        end;
      end;
    leSymbol_Library:
      begin
        NextNoSpace;
        case Id of
          leOf: Exclude(fBooleanDirectives, bdSymbol_Library);
          leOnDirective: Include(fBooleanDirectives, bdSymbol_Library);
        else Id := leUnknownDirective;
        end;
      end;
    leSymbol_Deprecated:
      begin
        NextNoSpace;
        case Id of
          leOf: Exclude(fBooleanDirectives, bdSymbol_Deprecated);
          leOnDirective: Include(fBooleanDirectives, bdSymbol_Deprecated);
        else Id := leUnknownDirective;
        end;
      end;
    leUnit_Deprecated:
      begin
        NextNoSpace;
        case Id of
          leOf: Exclude(fBooleanDirectives, bdUnit_Deprecated);
          leOnDirective: Include(fBooleanDirectives, bdUnit_Deprecated);
        else Id := leUnknownDirective;
        end;
      end;
    leUnit_Library:
      begin
        NextNoSpace;
        case Id of
          leOf: Exclude(fBooleanDirectives, bdUnit_Library);
          leOnDirective: Include(fBooleanDirectives, bdUnit_Library);
        else Id := leUnknownDirective;
        end;
      end;
    leUnit_Platform:
      begin
        NextNoSpace;
        case Id of
          leOf: Exclude(fBooleanDirectives, bdUnit_Platform);
          leOnDirective: Include(fBooleanDirectives, bdUnit_Platform);
        else Id := leUnknownDirective;
        end;
      end;
  else Id := leUnknownDirective;
  end;
  if Id <> leUnknownDirective then
    CheckCommaOrEnd;
end;

procedure TmwDelphiDirectiveParser.leWarningsHandler;
begin
  NextNoSpace;
  case Id of
    leOf: Exclude(fBooleanDirectives, bdWarnings);
    leOnDirective: Include(fBooleanDirectives, bdWarnings);
  else Id := leUnknownDirective;
  end;
  if Id <> leUnknownDirective then
    CheckCommaOrEnd;
end;

procedure TmwDelphiDirectiveParser.leWeakPackageUnitHandler;
begin
  NextNoSpace;
  case Id of
    leOf: Exclude(fBooleanDirectives, bdWeakPackageUnit);
    leOnDirective: Include(fBooleanDirectives, bdWeakPackageUnit);
  else Id := leUnknownDirective;
  end;
  if Id <> leUnknownDirective then
    CheckCommaOrEnd;
end;

procedure TmwDelphiDirectiveParser.leWHandler;
begin
  Next;
  case Id of
    leMinus: Exclude(fBooleanDirectives, bdStackFrames);
    lePlus: Include(fBooleanDirectives, bdStackFrames);
  else Id := leUnknownDirective;
  end;
  if Id <> leUnknownDirective then
    CheckCommaOrEnd;
end;

procedure TmwDelphiDirectiveParser.leWriteableConstHandler;
begin
  NextNoSpace;
  case Id of
    leOf: Exclude(fBooleanDirectives, bdWriteableConst);
    leOnDirective: Include(fBooleanDirectives, bdWriteableConst);
  else Id := leUnknownDirective;
  end;
  if Id <> leUnknownDirective then
    CheckCommaOrEnd;
end;

procedure TmwDelphiDirectiveParser.leXHandler;
begin
  Next;
  case Id of
    leMinus: Exclude(fBooleanDirectives, bdExtendedSyntax);
    lePlus: Include(fBooleanDirectives, bdExtendedSyntax);
  else Id := leUnknownDirective;
  end;
  if Id <> leUnknownDirective then
    CheckCommaOrEnd;
end;

procedure TmwDelphiDirectiveParser.leYDHandler;
begin
  Include(fBooleanDirectives, bdDefinitionInfo);
  CheckCommaOrEnd;
end;

procedure TmwDelphiDirectiveParser.leYHandler;
begin
  Next;
  case Id of
    leMinus: Exclude(fBooleanDirectives, bdDefinitionInfo);
    lePlus: Include(fBooleanDirectives, bdDefinitionInfo);
  else Id := leUnknownDirective;
  end;
  if Id <> leUnknownDirective then
    CheckCommaOrEnd;
end;

procedure TmwDelphiDirectiveParser.leZHandler;
var
  Temp: Integer;
begin
  Next;
  case Id = leNumber of
    True:
      begin
        Temp := StrToInt(Token);
        case Temp of
          1, 2, 4: SetMinEnumSize(Temp);
        else Id := leUnknownDirective;
        end;
      end;
  else Id := leUnknownDirective;
  end;
  if Id <> leUnknownDirective then
  begin
    CheckCommaOrEnd;
  end;
end;

procedure TmwDelphiDirectiveParser.LinkObjectFile(FileName: AnsiString);
begin

end;

procedure TmwDelphiDirectiveParser.NextToken;
begin
  Start := Run;
  ExId := leUnknown;
  case Run < TheEnd of
    True:
      begin
        case Range of
          drNormal,
            drAnsiDirective,
            drBorlandDirective:
            case Buf[Run] of
              #0..#9: SpaceHandler;
              #10: LFHandler;
              #11, #12: SpaceHandler;
              #13: CRHandler;
              #14..#32: SpaceHandler;
              '!': ExclamHandler;
              '"': DoubleQuoteHandler;
              '#': SharpHandler;
              '$': DollarHandler;
              '%': PercentHandler;
              '&': AmpersandHandler;
              '''': ApostropheHandler;
              '(': RoundOpenHandler;
              ')': RoundCloseHandler;
              '*': AsteriskHandler;
              '+': PlusHandler;
              ',': CommaHandler;
              '-': MinusHandler;
              '.': PeriodHandler;
              '/': SlashHandler;
              '0'..'9': NumberHandler;
              ':': ColonHandler;
              ';': SemicolonHandler;
              '<': LessHandler;
              '=': EqualHandler;
              '>': GreaterHandler;
              '?': QuestionHandler;
              '@': AtHandler;
              'A'..'Z': IdentifierHandler;
              '[': SquareOpenHandler;
              '\': BackslashHandler;
              ']': SquareCloseHandler;
              '^': AsciiCircumHandler;
              '_': IdentifierHandler;
              '`': QuoteLeftHandler;
              'a'..'z': IdentifierHandler;
              '{': CurlyOpenHandler;
              '|': BarHandler;
              '}': CurlyCloseHandler;
              '~': AsciiTildeHandler;
            else
              case InInternationalIdentifiers[Buf[Run]] of
                True: InternationalIdentifierHandler;
                False:
                  case (Buf[Run] > #191) and (Run + UTF8Width[Buf[Run]] <= TheEnd) of
                    True: InternationalIdentifierHandler;
                    False: UnknownHandler;
                  end;
              end;
            end;
          drAssemblerReference: NextAssemblerReference;
          drAnsiComment: NextAnsiComment;
          drBorlandComment: NextBorlandComment;
        end;
        case Id of
          leBeginOfAnsiDirective,
            leBeginOfBorlandDirective:
            MainDirectiveHandler;
        end;
      end;
    False: Id := leAtEnd;
  end
end;

procedure TmwDelphiDirectiveParser.Parse;
begin
  while Id <> leAtEnd do
    NextToken;
end;

procedure TmwDelphiDirectiveParser.SkipUntilEndIf;
var
  Condition: Boolean;
begin
  Condition := False;
  while Id <> leAtEnd do
    case Id of
      leEndIf:
        begin
          if DirectiveStack.Count > 0 then
            case Integer(DirectiveStack.Peek) of
              leElseDirective, leIfDef, leIfNDef, leIfOpt: Condition := True;
            end;
          case Condition of
            True:
              begin
                DirectiveStack.Pop;
                FindDirectiveEnd;
                Break;
              end;
            False:
              begin
                if Assigned(fOnLexerError) then
                  fOnLexerError(Self, 'EndIf without previous IfDef, Else, IfNDef or IfOpt directive');
              end;
          end;
        end;
      leIfDef, leIfNDef, leIfOpt:
        begin
          DirectiveStack.Push(TObject(Id));
          FindDirectiveEnd;
          Next;
          SkipUntilEndIf;
        end;
    else Next;
    end;
end;

procedure TmwDelphiDirectiveParser.SkipUntilIfEnd;
var
  Condition: Boolean;
begin
  Condition := False;
  while Id <> leAtEnd do
    case Id of
      leIfEnd:
        begin
          if DirectiveStack.Count > 0 then
            case Integer(DirectiveStack.Peek) of
              leElseIf, leIfDirective: Condition := True;
            end;
          case Condition of
            True:
              begin
                DirectiveStack.Pop;
                FindDirectiveEnd;
                Break;
              end;
            False:
              begin
                if Assigned(fOnLexerError) then
                  fOnLexerError(Self, 'IfEnd without previous If or ElseIf directive');
              end;
          end;
        end;
      leIfDirective:
        begin
          DirectiveStack.Push(TObject(Id));
          FindDirectiveEnd;
          Next;
          SkipUntilIfEnd;
        end;
    else Next;
    end;
end;

procedure TmwDelphiDirectiveParser.SkipUntilElseIfOrIfEnd;
var
  Condition: Boolean;
begin
  Condition := False;
  while Id <> leAtEnd do
    case Id of
      leIfEnd, leElseIf,
      leElseDirective:
        begin
          if DirectiveStack.Count > 0 then
            case Integer(DirectiveStack.Peek) of
              leElseIf, leIfDirective: Condition := True;
            end;
          case Condition of
            True:
              begin
                DirectiveStack.Pop;
                if Id <> leIfEnd then
                  DirectiveStack.Push(TObject(Id));
                FindDirectiveEnd;
                Break;
              end;
            False:
              begin
                if Assigned(fOnLexerError) then
                  fOnLexerError(Self, 'IfEnd/ElseIf without previous If or ElseIf directive');
              end;
          end;
        end;
      leIfDirective:
        begin
          DirectiveStack.Push(TObject(Id));
          FindDirectiveEnd;
          Next;
          SkipUntilIfEnd;
        end;
    else Next;
    end;
end;

procedure TmwDelphiDirectiveParser.SkipUntilElseOrEndIf;
var
  Condition: Boolean;
begin
  Condition := False;
  while Id <> leAtEnd do
    case Id of
      leElseDirective, leEndIf:
        begin
          if DirectiveStack.Count > 0 then
            case Integer(DirectiveStack.Peek) of
              leElseDirective, leIfDef, leIfNDef, leIfOpt: Condition := True;
            end;
          case Condition of
            True:
              begin
                DirectiveStack.Pop;
                 if Id = leElseDirective then
                  DirectiveStack.Push(TObject(Integer(leElseDirective)));
                FindDirectiveEnd;
                Break;
              end;
            False:
              begin
                if Assigned(fOnLexerError) then
                  fOnLexerError(Self, 'Else/EndIf without previous IfDef, Else, IfNDef or IfOpt directive');
              end;
          end;
        end;
      leIfDef, leIfNDef, leIfOpt:
        begin
          DirectiveStack.Push(TObject(Id));
          FindDirectiveEnd;
          Next;
          SkipUntilEndIf;
        end;
    else Next;
    end;
end;

end.

